%KM-SUB-Skin-Clothing model to investigate the RH dependence of squalene
%ozonolysis products.

%This code outputs the results shown by the solid lines in Figure 6 of
%Lakey et al. Environ. Sci. Atmos., 2023,3, 49

clear all
warning off all
global K_clothing kg kbr S_clothing V_room Porosity
global Clothes_fraction_air Clothes_fraction_fibres
global NCom L_layers hm delta_clothes Dg ratio_p
global V_clothes A ratio_s AER Production_O3
global Travel_distance delta_oil delta_sc delta_ve
global gap_length oil_layers sc_layers ve_layers K
global sigma_1 W_vel delta_x_1 a0_1 kd_1 ka
global Db_oil Db_oil_2 Db_sc Db_ve V_gap V_mfp V_oil V_sc V_ve

%Note that species and layer numbers are labelled in the
%Squalene_chem_clothing_F script (Production and Loss rates)

%%%%%%%--------------Parameters-----------------%%%%%%%%%%
NCom=69; % Number of species in the model
NA=6.02e23; %Avogadro's number (mol-1)
R=8.31e7; % Gas constant (g cm2 s-2 K-1 mol-1) 
T=273+23; % Temperature (K)
R_H=82.05746; %Gas constant (cm3 atm K-1 mol-1)

%%%%%%%---Parameters for species that can partition to the gas phase----%%%%%%%%%%
Mx(1)=154; % Molar mass of 4-MOD (g mol-1)
Mx(2)=168; % Molar mass of 4-MON (g mol-1)
Mx(3)=100; % Molar mass of 4-OPA (g mol-1)
Mx(4)=126; % Molar mass of 6-MHO (g mol-1)
Mx(6)=116; % Molar mass of CA1 (g mol-1)
Mx(7)=102; % Molar mass of CA2 (g mol-1)
Mx(20)=194; % Molar mass of Geranyl acetone (g mol-1)
Mx(21)=18; % Molar mass of H2O (g mol-1)
Mx(22)=74; % Molar mass of HK1 (g mol-1)
Mx(23)=116; % Molar mass of HK2 (g mol-1)
Mx(24)=142; % Molar mass of HK3 (g mol-1)
Mx(26)=48; % Molar mass of O3 (g mol-1)
Mx(46)=58; % Molar mass of acetone (g mol-1)
Mx(47)=86; % Molar mass of 1,4 butanedial (g mol-1)

for i=[1,2,3,4,6,7,20,21,22,23,24,26,46,47]
    Dens(i)=1; % Assumed density of the compounds (g cm-3)
    V_w(i)=Mx(i)/NA/Dens(i); % Volume of one molecule (cm3)
    delta_x(i)=V_w(i)^(1/3); % Geometrical diameter of one molecule (cm)
    sigma(i)=delta_x(i)^2; % Effective molecular cross section of one molecule (cm2)
    W(i)=sqrt(8*R*T/pi/Mx(i)); % Mean thermal velocity (cm s-1)
end
% Note: There are 21 gas-phase compounds of which 14 can partition between
% the gas-phase and skin/clothing. The following code changes the
% assignment of values from the species' condensed phase number to their
% gas-phase number

for i=1:4
sigma_1(i)=sigma(i); % Effective molecular cross section of one molecule (cm2)
W_vel(i)=W(i); % Mean thermal velocity (cm s-1)
delta_x_1(i)=delta_x(i); % Geometrical diameter of one molecule (cm)
end
for i=5:6
sigma_1(i)=sigma(i+1); % Effective molecular cross section of one molecule (cm2)
W_vel(i)=W(i+1); % Mean thermal velocity (cm s-1)
delta_x_1(i)=delta_x(i+1); % Geometrical diameter of one molecule (cm)
end
for i=14:18
sigma_1(i)=sigma(i+6); % Effective molecular cross section of one molecule (cm2)
W_vel(i)=W(i+6); % Mean thermal velocity (cm s-1)
delta_x_1(i)=delta_x(i+6); % Geometrical diameter of one molecule (cm)
end
for i=19
sigma_1(i)=sigma(i+7); % Effective molecular cross section of one molecule (cm2)
W_vel(i)=W(i+7); % Mean thermal velocity (cm s-1)
delta_x_1(i)=delta_x(i+7); % Geometrical diameter of one molecule (cm)
end
for i=20:21
sigma_1(i)=sigma(i+26); % Effective molecular cross section of one molecule (cm2)
W_vel(i)=W(i+26); % Mean thermal velocity (cm s-1)
delta_x_1(i)=delta_x(i+26); % Geometrical diameter of one molecule (cm)
end

%%%%%%----------Geometric factors----------------------%%%%%%%

L_layers=10; %Number of layers in the clothes
oil_layers=5;%Number of bulk layers in the skin oil layer
sc_layers=3; %Number of bulk layers in the stratum corneum
ve_layers=3; %Number of bulk layers in the viable epidermis

Thickness_of_the_clothing=0.1; %Thickness of the clothing (cm)
Thickness_of_the_oil=1e-4; %Thickness of the skin oil (cm)
Thickness_of_the_sc=25e-4; %Thickness of the stratum corneum (cm)
Thickness_of_the_ve=100e-4; %Thickness of the viable epidermis (cm)
gap_length=0.1; %Thickness of the gap between the clothes and the skin oil (cm)

delta_clothes=Thickness_of_the_clothing./L_layers; %Thickness of a layer of the clothing (cm)
delta_oil=Thickness_of_the_oil./oil_layers; %Thickness of a layer of the skin oil (cm)
delta_sc=Thickness_of_the_sc./sc_layers; %Thickness of a layer of the stratum corneum (cm)
delta_ve=Thickness_of_the_ve./ve_layers; %Thickness of a layer of the viable epidermis (cm)

A=1; %Surface area being tested (cm2)

V_clothes=delta_clothes*A; %Volume of a layer of the clothing (cm3)
V_gap=A*gap_length; %Volume of the gap between the clothes and the skin oil (cm3)
V_oil=A*delta_oil; %Volume of a layer of the skin oil (cm3)
V_sc=A*delta_sc; %Volume of a layer of the stratum corneum (cm3)
V_ve=A*delta_ve; %Volume of a layer of the viable epidermis (cm3)

%%%%%%------------Kinetic parameters----------------------%%%%%%%

for i=1:21
    a0_1(i)=0.63; %Surface mass accommodation of all species on an adsorbate free surface
    Td(i)=1e-9; %Desorption lifetime of all species on the skin surface (s)
    kd_1(i)=1/Td(i); %Desorption rate of all species from the skin surface (s-1)
    ka(i)=(a0_1(i)*W_vel(i))/4; % Adsorption rate of all species on an adsorbate free surface (cm s-1)
end

Porosity=0.6; %Clothing porosity
Clothes_fraction_air=0.75; % Fraction of the clothing which consists of air
Clothes_fraction_fibres=0.01;%Fraction of the clothing which consists of skin oil and other substances

%%%%%%---------------Diffusion coefficients---------------%%%%%%%
Db_sc=zeros(NCom); Db_ve=zeros(NCom); Db_oil=zeros(NCom); Dg=zeros(NCom);

Dg(1)=0.06;% Gas-phase diffusion coefficient of 4-MOD (cm2 s-1)
Dg(2)=0.06;% Gas-phase diffusion coefficient of 4-MON (cm2 s-1)
Dg(3)=0.08;% Gas-phase diffusion coefficient of 4-OPA (cm2 s-1)
Dg(4)=0.07;% Gas-phase diffusion coefficient of 6-MHO (cm2 s-1)
Dg(6)=0.08;% Gas-phase diffusion coefficient of CA1 (cm2 s-1)
Dg(7)=0.09;% Gas-phase diffusion coefficient of CA2 (cm2 s-1)
Dg(20)=0.05;% Gas-phase diffusion coefficient of Geranyl acetone (cm2 s-1)
Dg(21)=0.20;% Gas-phase diffusion coefficient of H2O (cm2 s-1)
Dg(22)=0.10;% Gas-phase diffusion coefficient of HK1 (cm2 s-1)
Dg(23)=0.08;% Gas-phase diffusion coefficient of HK2 (cm2 s-1)
Dg(24)=0.07;% Gas-phase diffusion coefficient of HK3 (cm2 s-1)
Dg(26)=0.16;% Gas-phase diffusion coefficient of O3 (cm2 s-1)
Dg(46)=0.11;% Gas-phase diffusion coefficient of acetone (cm2 s-1)
Dg(47)=0.09;% Gas-phase diffusion coefficient of 1,4 butanedial (cm2 s-1)

for i=1:48
    Db_oil(i)=1e-7; %Diffusion coefficient in the skin oil unless specified below (cm2 s-1)
end
Db_oil(1)=9.06e-7;% Diffusion coefficient in the skin oil for 4MOD (cm2 s-1)
Db_oil(2)=7.62e-7;% Diffusion coefficient in the skin oil for 4MON (cm2 s-1)
Db_oil(3)=1.56e-6;% Diffusion coefficient in the skin oil for 4OPA (cm2 s-1)
Db_oil(4)=1.09e-6;% Diffusion coefficient in the skin oil for 6MHO (cm2 s-1)
Db_oil(6)=1e-6;% Diffusion coefficient in the skin oil for CA1 (cm2 s-1)
Db_oil(7)=1e-6;% Diffusion coefficient in the skin oil for CA2 (cm2 s-1)
Db_oil(20)=5.88e-7;% Diffusion coefficient in the skin oil for Geranyl acetone (cm2 s-1)
Db_oil(21)=5e-6;% Diffusion coefficient in the skin oil for H2O (cm2 s-1)
Db_oil(22)=1e-6;% Diffusion coefficient in the skin oil for HK1 (cm2 s-1)
Db_oil(23)=1e-6;% Diffusion coefficient in the skin oil for HK2 (cm2 s-1)
Db_oil(24)=1e-6;% Diffusion coefficient in the skin oil for HK3 (cm2 s-1)
Db_oil(26)=3.6e-6;% Diffusion coefficient in the skin oil for O3 (cm2 s-1)
Db_oil(46)=2.08e-6;% Diffusion coefficient in the skin oil for acetone (cm2 s-1)
Db_oil(47)=1.81e-6;% Diffusion coefficient in the skin oil for butanedial (cm2 s-1)

% Note: There are 21 gas-phase compounds of which 14 can partition between
% the gas-phase and skin/clothing. The following code changes the
% assignment of values from the species' condensed phase number to their
% gas-phase number

for i=1:4
Db_oil_2(i)=Db_oil(i); % Diffusion coefficient in the skin oil (cm2 s-1)
end
for i=5:6
Db_oil_2(i)=Db_oil(i+1); % Diffusion coefficient in the skin oil (cm2 s-1)
end
for i=14:18
Db_oil_2(i)=Db_oil(i+6); % Diffusion coefficient in the skin oil (cm2 s-1)
end
for i=19
Db_oil_2(i)=Db_oil(i+7); % Diffusion coefficient in the skin oil (cm2 s-1)
end
for i=20:21
Db_oil_2(i)=Db_oil(i+26); % Diffusion coefficient in the skin oil (cm2 s-1)
end

for i=1:48
    Db_sc(i)=7.3e-18; %Diffusion coefficient in the stratum corneum unless specified below (cm2 s-1)
    Db_ve(i)=3.4e-13; %Diffusion coefficient in the viable epidermis unless specified below (cm2 s-1)
end
Db_sc(1)=3.1e-11; %Diffusion coefficient in the stratum corneum for 4MOD (cm2 s-1)
Db_sc(2)=2.1e-11; %Diffusion coefficient in the stratum corneum for 4MON (cm2 s-1)
Db_sc(3)=5.9e-12; %Diffusion coefficient in the stratum corneum for 4OPA (cm2 s-1)
Db_sc(4)=9.7e-11; %Diffusion coefficient in the stratum corneum for 6MHO (cm2 s-1)
Db_sc(6)=5.3e-12; %Diffusion coefficient in the stratum corneum for CA1 (cm2 s-1)
Db_sc(7)=8.5e-12; %Diffusion coefficient in the stratum corneum for CA2 (cm2 s-1)
Db_sc(8)=9.5e-12; %Diffusion coefficient in the stratum corneum for CA9 (cm2 s-1)
Db_sc(20)=1.3e-10; %Diffusion coefficient in the stratum corneum for Geranyl acetone (cm2 s-1)
Db_sc(21)=5e-8; %Diffusion coefficient in the stratum corneum for H2O (cm2 s-1)
Db_sc(22)=1.3e-11; %Diffusion coefficient in the stratum corneum for HK1 (cm2 s-1)
Db_sc(23)=1.9e-12; %Diffusion coefficient in the stratum corneum for HK2 (cm2 s-1)
Db_sc(24)=3.5e-11; %Diffusion coefficient in the stratum corneum for HK3 (cm2 s-1)
Db_sc(25)=8.3e-11; %Diffusion coefficient in the stratum corneum for HK5 (cm2 s-1)
Db_sc(26)=1e-8; %Diffusion coefficient in the stratum corneum for O3 (cm2 s-1)
Db_sc(28)=7.2e-11; %Diffusion coefficient in the stratum corneum for Product A (cm2 s-1)
Db_sc(29)=7.2e-11; %Diffusion coefficient in the stratum corneum for Product B (cm2 s-1)
Db_sc(30)=1.9e-11; %Diffusion coefficient in the stratum corneum for Product C (cm2 s-1)
Db_sc(31)=3.4e-13; %Diffusion coefficient in the stratum corneum for Product D (cm2 s-1)
Db_sc(42)=7.3e-18; %Diffusion coefficient in the stratum corneum for Squalene (cm2 s-1)
Db_sc(43)=1.5e-13; %Diffusion coefficient in the stratum corneum for TOP (cm2 s-1)
Db_sc(44)=1.9e-15; %Diffusion coefficient in the stratum corneum for TOT (cm2 s-1)
Db_sc(45)=1.5e-11; %Diffusion coefficient in the stratum corneum for TTT (cm2 s-1)
Db_sc(46)=4.6e-11; %Diffusion coefficient in the stratum corneum for acetone (cm2 s-1)
Db_sc(47)=9.8e-12; %Diffusion coefficient in the stratum corneum for 1,4 butanedial (cm2 s-1)

Db_ve(1)=1.0e-6; %Diffusion coefficient in the viable epidermis for 4MOD (cm2 s-1)
Db_ve(2)=1.0e-6; %Diffusion coefficient in the viable epidermis for 4MON (cm2 s-1)
Db_ve(3)=3.0e-6; %Diffusion coefficient in the viable epidermis for 4OPA (cm2 s-1)
Db_ve(4)=8.6e-7; %Diffusion coefficient in the viable epidermis for 6MHO (cm2 s-1)
Db_ve(6)=2.6e-6; %Diffusion coefficient in the viable epidermis for CA1 (cm2 s-1)
Db_ve(7)=2.8e-6; %Diffusion coefficient in the viable epidermis for CA2 (cm2 s-1)
Db_ve(8)=3.8e-9; %Diffusion coefficient in the viable epidermis for CA9 (cm2 s-1)
Db_ve(20)=9.7e-8; %Diffusion coefficient in the viable epidermis for Geranyl acetone (cm2 s-1)
Db_ve(21)=5e-6; %Diffusion coefficient in the viable epidermis for H2O (cm2 s-1)
Db_ve(22)=3.6e-6; %Diffusion coefficient in the viable epidermis for HK1 (cm2 s-1)
Db_ve(23)=2.9e-6; %Diffusion coefficient in the viable epidermis for HK2 (cm2 s-1)
Db_ve(24)=1.1e-6; %Diffusion coefficient in the viable epidermis for HK3 (cm2 s-1)
Db_ve(25)=1.6e-7; %Diffusion coefficient in the viable epidermis for HK5 (cm2 s-1)
Db_ve(26)=1e-6; %Diffusion coefficient in the viable epidermis for O3 (cm2 s-1)
Db_ve(28)=1.4e-7; %Diffusion coefficient in the viable epidermis for Product A (cm2 s-1)
Db_ve(29)=1.4e-7; %Diffusion coefficient in the viable epidermis for Product B (cm2 s-1)
Db_ve(30)=1.0e-8; %Diffusion coefficient in the viable epidermis for Product C (cm2 s-1)
Db_ve(31)=5.2e-10; %Diffusion coefficient in the viable epidermis for Product D (cm2 s-1)
Db_ve(42)=3.4e-13; %Diffusion coefficient in the viable epidermis for Squalene (cm2 s-1)
Db_ve(43)=2.7e-10; %Diffusion coefficient in the viable epidermis for TOP (cm2 s-1)
Db_ve(44)=1.3e-11; %Diffusion coefficient in the viable epidermis for TOT (cm2 s-1)
Db_ve(45)=5.4e-9; %Diffusion coefficient in the viable epidermis for TTT (cm2 s-1)
Db_ve(46)=3.9e-6; %Diffusion coefficient in the viable epidermis for acetone (cm2 s-1)
Db_ve(47)=3.2e-6; %Diffusion coefficient in the viable epidermis for 1,4 butanedial (cm2 s-1)

for i=[1,2,3,4,6,7,20,21,22,23,24,26,46,47]
    mfp(i)=3*Dg(i)./W(i); %Mean free path (cm)
    V_mfp_2(i)=A.*mfp(i); %Volume of the mean free path (cm3)
end

% Note: There are 21 gas-phase compounds of which 14 can partition between
% the gas-phase and skin/clothing. The following code changes the
% assignment of values from the species' condensed phase number to their
% gas-phase number
for i=1:4
    V_mfp(i)=V_mfp_2(i); %Volume of the mean free path (cm3)
end
for i=5:6
    V_mfp(i)=V_mfp_2(i+1); %Volume of the mean free path (cm3)
end
for i=14:18
    V_mfp(i)=V_mfp_2(i+6); %Volume of the mean free path (cm3)
end
for i=19
    V_mfp(i)=V_mfp_2(i+7); %Volume of the mean free path (cm3)
end
for i=20:21
    V_mfp(i)=V_mfp_2(i+26); %Volume of the mean free path (cm3)
end

Travel_distance=1.2; %Boundary layer thickness (cm)
hm(26)=Dg(26)/Travel_distance; %mass- transfer coefficient associated with flux from room air to the clothing for ozone (cm s-1)
for i=[1,2,3,4,6,7,20,21,22,23,24,26,46,47]
    hm(i)=hm(26)/Dg(26)^(2/3)*Dg(i)^(2/3); %mass- transfer coefficient associated with flux from room air to the clothing for other species (cm s-1)
end

%%%%%%%--------Partioning coefficients--------%%%%%
%Partitioning coefficients for clothing (unitless):

K_clothing(1)=2.55e6; % 4MOD
K_clothing(2)=8.51e5; % 4MON
K_clothing(3)=9.73e3; % 4OPA
K_clothing(4)=1.46e4; %6MHO
K_clothing(6)=4.44e5; %CA1
K_clothing(7)=6.59e5; %CA2
K_clothing(20)=7.89e6; %Geranyl acetone
K_clothing(21)=10.14; % H2O
K_clothing(22)=2.88e4; % HK1
K_clothing(23)=1.28e6; % HK2
K_clothing(24)=9.75e5; % HK3
K_clothing(26)=15.6; % O3
K_clothing(46)=32.43; % acetone
K_clothing(47)=3.24e3; %1,4 butanedial

%Partitioning coefficients for skin oil (unitless)
K(1)=2.68e-1*R_H*T; % 4MOD
K(2)=4.74e-1*R_H*T; % 4MON
K(3)=7.47e-3*R_H*T; % 4OPA
K(4)=8.29e-2*R_H*T; % 6MHO
K(6)=0.37*R_H*T; % CA1
K(7)=0.54*R_H*T; % CA2
K(20)=5.77*R_H*T; % Geranyl acetone
K(21)=K_clothing(21); % H2O
K(22)=0.024*R_H*T; % HK1
K(23)=1.05*R_H*T; % HK2
K(24)=0.80*R_H*T; % HK3
K(26)=K_clothing(26); % O3
K(46)=5.44e-4*R_H*T; % acetone
K(47)=2.39e-3*R_H*T; %1,4 butanedial

%%%%%%%--------Reaction rate coefficients--------%%%%%
%Overall condensed-phase reaction rate coefficients:
k1=2.2e-17; % O3 + double bond → ½ (BR1 primary CI + BR1 secondary carbonyl + BR2 secondary CI + BR2 primary carbonyl) reaction rate coefficient (cm3 s-1)
k2=5.7e-19; % CI + H2O → Carbonyl + H2O2 reaction rate coefficient (cm3 s-1)							
k3=9.61e-17; % CI + O3 → Carbonyl reaction rate coefficient (cm3 s-1) 					
k4=0.4392; % CI → Carboxylic acid or Hydroxy ketone reaction rate coefficient (s-1)						
k5=1e-20; % CI + Carbonyl → Secondary ozonide reaction rate coefficient (cm3 s-1)
k6=1.8e-20; % CI + Carboxylic acid → Hydroperoxide reaction rate coefficient (cm3 s-1)							

%Note k6 is reported as 2.7e-20 cm3 s-1 in Lakey et al. but was actually set to
%1.8e-20 cm3 s-1 for Figure S5. The difference in results is very small but noticeable.

%Overall gas-phase reaction rate coefficients:
k8=4.0e-16; % O3 + double bond → ½ (BR1 primary CI + BR1 secondary carbonyl + BR2 secondary CI + BR2 primary carbonyl) reaction rate coefficient (cm3 s-1)
k9=4.37e-16; % CI + H2O → Carbonyl + H2O2 reaction rate coefficient (cm3 s-1)
k10=2.41e-14; % CI + O3 → Carbonyl reaction rate coefficient (cm3 s-1)
k11=1.11; % CI → Carboxylic acid or Hydroxy ketone reaction rate coefficient (s-1)

%Branching ratios in the condensed phase (note, equivalent values in the gas-phase would be 1)
ratio_p=0.2; %Branching ratio 1
ratio_s=2-ratio_p; %Branching ratio 2

%Condensed-phase reaction rate coefficients (note kbr(x) is the reaction rate coefficient for Reaction x)
kbr(1)=k1*6;
kbr(2)=k1;
kbr(3)=k1*2;
kbr(4)=k1*3;
kbr(5)=k1;
kbr(6)=k1;
kbr(7)=k1*2;
kbr(8)=k1*4;
kbr(9)=k1*2;
kbr(10)=k1*3;
kbr(11)=k1*5;
kbr(12)=k1*4;
kbr(13)=k1;
kbr(14)=k1*2;
kbr(15)=k1*3;
kbr(16)=k2;
kbr(17)=k3;
kbr(18)=k4;
kbr(19)=k5;
kbr(20)=k2;
kbr(21)=k3;
kbr(22)=k4;
kbr(23)=k5;
kbr(24)=k2;
kbr(25)=k3;
kbr(26)=k4;
kbr(27)=k5;
kbr(28)=k2;
kbr(29)=k3;
kbr(30)=k4;
kbr(31)=k5;
kbr(32)=k2;
kbr(33)=k3;
kbr(34)=k4;
kbr(35)=k5;
kbr(36)=k2;
kbr(37)=k3;
kbr(38)=k4;
kbr(39)=k5;
kbr(40)=k2;
kbr(41)=k3;
kbr(42)=k4;
kbr(43)=k5;
kbr(44)=k2;
kbr(45)=k3;
kbr(46)=k4;
kbr(47)=k5;
kbr(48)=k5;
kbr(49)=k5;
kbr(50)=k5;
kbr(51)=k5;
kbr(52)=k5;
kbr(53)=2*k5;
kbr(54)=2*k5;
kbr(55)=2*k5;
kbr(56)=2*k5;
kbr(57)=2*k5;
kbr(58)=2*k5;
kbr(59)=k5;
kbr(60)=k5;
kbr(61)=0;%doen't exist
kbr(62)=0;%doen't exist
kbr(63)=0;%doen't exist
kbr(64)=2*k5;
kbr(65)=k5;
kbr(66)=0;%doen't exist
kbr(67)=k5;
kbr(68)=k1;
kbr(69)=k1;
kbr(70)=k1;
kbr(71)=k1;
kbr(72)=k1;
kbr(73)=k1*2;
kbr(74)=k1*3;
kbr(75)=k2;
kbr(76)=k3;
kbr(77)=k4;
kbr(78)=k5;
kbr(79)=k2;
kbr(80)=k3;
kbr(81)=k4;
kbr(82)=k5;
kbr(83)=k5;
kbr(84)=k5;
kbr(85)=k5;
kbr(86)=k5;
kbr(87)=k5;
kbr(88)=k5;
kbr(89)=k5;
kbr(90)=k6;
kbr(91)=k6;
kbr(92)=k6;
kbr(93)=k6;
kbr(94)=k6;
kbr(95)=k6;
kbr(96)=k6;
kbr(97)=k6;
kbr(98)=k6;
kbr(99)=k6;
kbr(100)=k1;
kbr(101)=k2;
kbr(102)=k3;
kbr(103)=k4;
kbr(104)=k5;
kbr(105)=k6;
kbr(106)=k5;

%Gas-phase reaction rate coefficients
kg(1)=k8*2;
kg(2)=k8;
kg(3)=k8;
kg(4)=k8;
kg(5)=k8;
kg(6)=k9;
kg(7)=k10;
kg(8)=k11;
kg(9)=k9;
kg(10)=k10;
kg(11)=k11;
kg(12)=k9;
kg(13)=k10;
kg(14)=k11;
kg(15)=k9;
kg(16)=k10;
kg(17)=k11;
kg(18)=k9;
kg(19)=k10;
kg(20)=k11;
kg(21)=k9;
kg(22)=k10;
kg(23)=k11;
kg(24)=k9;
kg(25)=k10;
kg(26)=k11;

%%%%%%%%----------------Building parameters------------------------%%%%%%%

AER=1/3600; %Air exchange rate (s-1)

S_clothing=2*1.7*1e4; %Total surface area of clothing (cm2)

V_room=28.5*1e6; %Total volume of the room (cm3)

Rel_humidity_range=linspace(0,100,100); %Range of relative humidities being tested (%)

for JJ=1:100

RH=Rel_humidity_range(JJ) %Relative humidity being tasted on run JJ (%)

Production_O3=33*2.46e10*AER; % O3 production (cm-3 s-1, maintains an O3 concentration of 33 ppb in the absence of people)


%%%%%%---------Initial concentrations------------------%%%%%%

y0=zeros(NCom*L_layers+105+oil_layers*48+sc_layers*48+48*ve_layers+48,1);

y0(NCom*L_layers+21+19)=33*2.46e10; %Initial gas-phase ozone concentration (cm-3)

XgH2O=RH*6.83e15;
y0(NCom*L_layers+21+15)=XgH2O; %Initial gas-phase H2O concentration (cm-3)

%----Clothes concentrations condensed phase----%
for m=1:L_layers
    y0(NCom*(m-1)+42)=5e18; %squalene concentration (cm-3)
    y0(NCom*(m-1)+5)=y0(NCom*(m-1)+42)*2.5;%concentration of acids (cm-3)
    y0(NCom*(m-1)+27)=2.5e18; %concentration of other double bonds (cm-3)
end

%----Skin oil concentrations----%
for m=1:oil_layers
  y0(NCom*L_layers+105+48*(m-1)+42)=1.3e20; %squalene concentration (cm-3)
  y0(NCom*L_layers+105+48*(m-1)+5)=2.5*y0(NCom*L_layers+105+48*(m-1)+42); %concentration of acids (cm-3)
  y0(NCom*L_layers+105+48*(m-1)+27)=6.5e19; %concentration of other double bonds (cm-3)  
end

%----Stratum corneum concentrations----%
for m=1:sc_layers
    y0(NCom*L_layers+105+oil_layers*48+48*(m-1)+42)=1.3e20/20; %squalene concentration (cm-3)
    y0(NCom*L_layers+105+oil_layers*48+48*(m-1)+5)=y0(NCom*L_layers+105+48*(m-1)+5); %concentration of acids (cm-3)
    y0(NCom*L_layers+105+oil_layers*48+48*(m-1)+27)=6.5e19; %concentration of other double bonds (cm-3)     
end

%***********Solve differential equations*********
n=2500;
t=logspace(-3,log10(4*3600),n);t=t'; %time being investigated (s)
AbsTol=1;
RelTol=1e-3;
options=odeset('AbsTol',AbsTol,'RelTol',RelTol);
[X,Y]=ode23tb(@Squalene_chem_clothing_F,t,y0); %solver

%**************Outputs**************************
%Gas-phase concentrations
fourh_MHO_output(JJ,1)=Y(n,NCom*L_layers+21+4)./2.46e10; %Concentration of 6-MHO after 4 hours
fourh_OPA_output(JJ,1)=Y(n,NCom*L_layers+21+3)./2.46e10; %Concentration of 4-OPA after 4 hours
fourh_acetone_output(JJ,1)=Y(n,NCom*L_layers+21+20)./2.46e10; %Concentration of acetone after 4 hours
fourh_butanedial_output(JJ,1)=Y(n,NCom*L_layers+21+21)./2.46e10; %Concentration of 1,4 butanedial after 4 hours
fourh_GA_output(JJ,1)=Y(n,NCom*L_layers+21+14)./2.46e10; %Concentration of Geranyl acetone after 4 hours
fourh_CA1_output(JJ,1)=Y(n,NCom*L_layers+21+5)./2.46e10; %Concentration of CA1 after 4 hours
fourh_CA2_output(JJ,1)=Y(n,NCom*L_layers+21+6)./2.46e10; %Concentration of CA2 after 4 hours
fourh_HK1_output(JJ,1)=Y(n,NCom*L_layers+21+16)./2.46e10; %Concentration of HK1 after 4 hours
fourh_O3_output(JJ,1)=Y(n,NCom*L_layers+21+19)./2.46e10; %Concentration of O3 after 4 hours
fourh_HK2_output(JJ,1)=Y(n,NCom*L_layers+21+17)./2.46e10; %Concentration of HK2 after 4 hours
fourh_HK3_output(JJ,1)=Y(n,NCom*L_layers+21+18)./2.46e10; %Concentration of HK3 after 4 hours
fourh_4MON_output(JJ,1)=Y(n,NCom*L_layers+21+2)./2.46e10; %Concentration of 6-MON after 4 hours
fourh_4MOD_output(JJ,1)=Y(n,NCom*L_layers+21+1)./2.46e10; %Concentration of 6-MOD after 4 hours

end

figure(1)
subplot(2,2,1)
plot(Rel_humidity_range(1,:),fourh_4MOD_output(:,1))
xlabel('Relative humidity (%)')
ylabel({'Gas phase conc.';'after 4 hours (ppb)'})
title('4MOD')

subplot(2,2,2)
plot(Rel_humidity_range(1,:),fourh_4MON_output(:,1))
xlabel('Relative humidity (%)')
ylabel({'Gas phase conc.';'after 4 hours (ppb)'})
title('4MON')

subplot(2,2,3)
plot(Rel_humidity_range(1,:),fourh_OPA_output(:,1))
xlabel('Relative humidity (%)')
ylabel({'Gas phase conc.';'after 4 hours (ppb)'})
title('4OPA')

subplot(2,2,4)
plot(Rel_humidity_range(1,:),fourh_MHO_output(:,1))
xlabel('Relative humidity (%)')
ylabel({'Gas phase conc.';'after 4 hours (ppb)'})
title('6MHO')

figure(2)
subplot(2,2,1)
plot(Rel_humidity_range(1,:),fourh_CA1_output(:,1))
xlabel('Relative humidity (%)')
ylabel({'Gas phase conc.';'after 4 hours (ppb)'})
title('CA1, levulinic acid')

subplot(2,2,2)
plot(Rel_humidity_range(1,:),fourh_CA2_output(:,1))
xlabel('Relative humidity (%)')
ylabel({'Gas phase conc.';'after 4 hours (ppb)'})
title('CA2, 4-oxobutanoic acid')

subplot(2,2,3)
plot(Rel_humidity_range(1,:),fourh_GA_output(:,1))
xlabel('Relative humidity (%)')
ylabel({'Gas phase conc.';'after 4 hours (ppb)'})
title('GA, Geranyl acetone')

subplot(2,2,4)
plot(Rel_humidity_range(1,:),fourh_butanedial_output(:,1))
xlabel('Relative humidity (%)')
ylabel({'Gas phase conc.';'after 4 hours (ppb)'})
title('1,4 butanedial')

figure(3)
subplot(2,2,1)
plot(Rel_humidity_range(1,:),fourh_HK1_output(:,1))
xlabel('Relative humidity (%)')
ylabel({'Gas phase conc.';'after 4 hours (ppb)'})
title('HK1, hydroxyacetone')

subplot(2,2,2)
plot(Rel_humidity_range(1,:),fourh_HK2_output(:,1))
xlabel('Relative humidity (%)')
ylabel({'Gas phase conc.';'after 4 hours (ppb)'})
title('HK2, 5-hydroxy-4-oxo-pentanal')

subplot(2,2,3)
plot(Rel_humidity_range(1,:),fourh_HK3_output(:,1))
xlabel('Relative humidity (%)')
ylabel({'Gas phase conc.';'after 4 hours (ppb)'})
title('HK3, 6-methyl-5-ene-2-oxoheptanol')

subplot(2,2,4)
plot(Rel_humidity_range(1,:),fourh_O3_output(:,1))
xlabel('Relative humidity (%)')
ylabel({'Gas phase conc.';'after 4 hours (ppb)'})
title('O3')

figure(4)
subplot(2,2,1)
plot(Rel_humidity_range(1,:),fourh_acetone_output(:,1))
xlabel('Relative humidity (%)')
ylabel({'Gas phase conc.';'after 4 hours (ppb)'})
title('acetone')
